static char rcsid[] = "$Id: cigar.c 224334 2021-06-21 18:27:54Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>		/* For isupper */

#include "cigar.h"
#include "assert.h"
#include "mem.h"
#include "complement.h"
#include "stage3hr.h"
#include "mdprint.h"


#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* print_extended_cigar */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* Cigar_compute_main */
#ifdef DEBUG3
#define debug3(x) x
#else
#define debug3(x)
#endif

/* Cigar_compute_supplemental */
#ifdef DEBUG4
#define debug4(x) x
#else
#define debug4(x)
#endif



static Genome_T genome;
static bool hide_soft_clips_p;
static bool cigar_extended_p = false;
static bool merge_samechr_p;
static bool md_lowercase_variant_p;
static bool sam_hardclip_use_S_p;
static bool sam_insert_0M_p;


void
Cigar_setup (Genome_T genome_in, bool cigar_extended_p_in, bool hide_soft_clips_p_in,
	     bool merge_samechr_p_in, bool md_lowercase_variant_p_in,
	     bool sam_hardclip_use_S_p_in, bool sam_insert_0M_p_in) {

  genome = genome_in;
  cigar_extended_p = cigar_extended_p_in;
  hide_soft_clips_p = hide_soft_clips_p_in;
  merge_samechr_p = merge_samechr_p_in;
  md_lowercase_variant_p = md_lowercase_variant_p_in;
  sam_hardclip_use_S_p = sam_hardclip_use_S_p_in;
  sam_insert_0M_p = sam_insert_0M_p_in;

  return;
}


/* Modified from print_cigar and assuming type == M */
static int
length_cigar_M (int stringlength, int querypos, int querylength,
		int hardclip_low, int hardclip_high, bool plusp, bool lastp) {
  int length = 0;
  int matchlength = 0;
  int startpos, endpos;
  int cliplength = 0;

  if (plusp == true) {
    debug1(printf("\nEntering length_cigar_M with stringlength %d, querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, plus\n",
		  stringlength,querypos,querylength,hardclip_low,hardclip_high));
    if (hardclip_low > querypos) { /* > not >= */
      startpos = hardclip_low;
      cliplength = hardclip_low;
    } else {
      startpos = querypos;
    }

    if (querylength - hardclip_high < querypos + stringlength) {
      endpos = querylength - hardclip_high;
      debug1(printf("  endpos %d = querylength %d - hardclip_high %d\n",endpos,querylength,hardclip_high));
    } else {
      endpos = querypos + stringlength;
      debug1(printf("  endpos %d = querypos %d + stringlength %d\n",endpos,querypos,stringlength));
    }

    debug1(printf("  new startpos %d, endpos %d, cliplength %d\n",startpos,endpos,cliplength));

    if (endpos >= startpos) {
      if (cliplength > 0) {
	debug1(printf("  Ignoring initial %dH\n",cliplength));
	/* FPRINTF(fp,"%dH",cliplength); */
      }
      matchlength = endpos - startpos;
      if (matchlength <= 0) {
	/* Skip */
      } else {
	/* type != 'E' */
	debug1(printf("  Adding length of  %d\n",matchlength));
	length += matchlength;	/* FPRINTF(fp,"%d%c",matchlength,type); */
      }
    }


    if (lastp == true) {
      /* cliplength = querypos + stringlength - endpos; */
      cliplength = querylength - endpos;
      if (cliplength > 0) {
	debug1(printf("  Ignoring final %dH\n",cliplength));
	/* FPRINTF(fp,"%dH",cliplength); */
      }
    }

  } else {
    debug1(printf("\nEntering length_cigar_M with stringlength %d, querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, minus\n",
		  stringlength,querypos,querylength,hardclip_low,hardclip_high));

    if (querylength - hardclip_low < querypos) {
      startpos = querylength - hardclip_low;
      cliplength = hardclip_low;
    } else {
      startpos = querypos;
    }

    if (hardclip_high >= querypos - stringlength) {
      endpos = hardclip_high;
      debug1(printf("  endpos %d = hardclip_high %d\n",endpos,hardclip_high));
    } else {
      endpos = querypos - stringlength;
      debug1(printf("  endpos %d = querypos %d - stringlength %d\n",endpos,querypos,stringlength));
    }

    debug1(printf("  new startpos %d, endpos %d, cliplength %d\n",startpos,endpos,cliplength));

    if (endpos <= startpos) {
      if (cliplength > 0) {
	debug1(printf("  Ignoring initial %dH\n",cliplength));
	/* FPRINTF(fp,"%dH",cliplength); */
      }
      matchlength = startpos - endpos;
      if (matchlength <= 0) {
	/* Skip */
      } else {
	/* type != 'E' */
	debug1(printf("  Adding length of %d\n",matchlength));
	length += matchlength;  /* FPRINTF(fp,"%d%c",matchlength,type); */
      }
    }


    if (lastp == true) {
      cliplength = endpos;
      if (cliplength > 0) {
	debug1(printf("  Ignoring final %dH\n",cliplength));
	/* FPRINTF(fp,"%dH",cliplength); */
      }
    }
  }

  return length;
}



/* Returns true if prints the requested type */
static bool
print_cigar (Filestring_T fp, char type, int stringlength, int querypos, int querylength,
	     int hardclip_low, int hardclip_high, bool plusp, bool lastp, int trimlength) {
  bool printp = false;
  int matchlength = 0;
  int startpos, endpos;
  int cliplength = 0;
  
  if (plusp == true) {
    debug1(printf("\nEntering print_cigar with type %c, stringlength %d, querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, plus\n",
		  type,stringlength,querypos,querylength,hardclip_low,hardclip_high));
    if (hardclip_low > querypos) { /* > not >= */
      startpos = hardclip_low;
      cliplength = hardclip_low;
    } else {
      startpos = querypos;
    }

    if (querylength - hardclip_high < querypos + stringlength) {
      endpos = querylength - hardclip_high;
      debug1(printf("  endpos %d = querylength %d - hardclip_high %d\n",endpos,querylength,hardclip_high));
    } else {
      endpos = querypos + stringlength;
      debug1(printf("  endpos %d = querypos %d + stringlength %d\n",endpos,querypos,stringlength));
    }

    debug1(printf("  new startpos %d, endpos %d, cliplength %d\n",startpos,endpos,cliplength));

    if (endpos >= startpos) {
      if (cliplength > 0) {
	debug1(printf("  Pushing initial %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
      matchlength = endpos - startpos;
      if (matchlength <= 0 && (type != 'M' || sam_insert_0M_p == false)) {
	/* Skip, except for 0M if desired */
      } else if (type != 'E') {
	debug1(printf("  Pushing %d%c\n",matchlength,type));
	FPRINTF(fp,"%d%c",matchlength,type);
	printp = true;
      } else if (matchlength == trimlength) {
	debug1(printf("  Pushing %dS\n",matchlength));
	FPRINTF(fp,"%dS",matchlength);
      } else {
	debug1(printf("  Pushing %dH because matchlength %d != trimlength %d\n",
		      matchlength,matchlength,trimlength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",matchlength);
	} else {
	  FPRINTF(fp,"%dH",matchlength);
	}
      }
    }


    if (lastp == true) {
      /* cliplength = querypos + stringlength - endpos; */
      cliplength = querylength - endpos;
      if (cliplength > 0) {
	debug1(printf("  Pushing final %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
    }

  } else {
    debug1(printf("\nEntering print_cigar with type %c, stringlength %d, querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, minus\n",
		  type,stringlength,querypos,querylength,hardclip_low,hardclip_high));

    if (querylength - hardclip_low < querypos) {
      startpos = querylength - hardclip_low;
      cliplength = hardclip_low;
    } else {
      startpos = querypos;
    }

    if (hardclip_high >= querypos - stringlength) {
      endpos = hardclip_high;
      debug1(printf("  endpos %d = hardclip_high %d\n",endpos,hardclip_high));
    } else {
      endpos = querypos - stringlength;
      debug1(printf("  endpos %d = querypos %d - stringlength %d\n",endpos,querypos,stringlength));
    }

    debug1(printf("  new startpos %d, endpos %d, cliplength %d\n",startpos,endpos,cliplength));

    if (endpos <= startpos) {
      if (cliplength > 0) {
	debug1(printf("  Pushing initial %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
      matchlength = startpos - endpos;
      if (matchlength <= 0 && (type != 'M' || sam_insert_0M_p == false)) {
	/* Skip, except for 0M if desired */
      } else if (type != 'E') {
	debug1(printf("  Pushing %d%c\n",matchlength,type));
	FPRINTF(fp,"%d%c",matchlength,type);
	printp = true;
      } else if (matchlength == trimlength) {
	debug1(printf("  Pushing %dS\n",matchlength));
	FPRINTF(fp,"%dS",matchlength);
      } else {
	debug1(printf("  Pushing %dH because matchlength %d != trimlength %d\n",
		      matchlength,matchlength,trimlength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",matchlength);
	} else {
	  FPRINTF(fp,"%dH",matchlength);
	}
      }
    }


    if (lastp == true) {
      cliplength = endpos;
      if (cliplength > 0) {
	debug1(printf("  Pushing final %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
    }
  }

  return printp;
}



/* Returns true if we printed an = or X entry */
/* Based on print_md_string */
static bool
print_extended_cigar (Filestring_T fp, char *genomicfwd_refdiff,
		      int stringlength, int querypos, int querylength,
		      int hardclip_low, int hardclip_high, bool plusp, bool lastp) {
  bool printp = false;
  int nmatches = 0, nmismatches = 0;
  int starti, endi, i;
  bool hardclip_end_p = false;
  int cliplength, endpos;

  if (plusp == true) {
    debug2(printf("\nEntering print_extended_cigar with querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, plus: %s ref\n",
		  querypos,querylength,hardclip_low,hardclip_high,genomicfwd_refdiff));
    if (hardclip_low == 0) {
      starti = 0;
      hardclip_end_p = true;
    } else if (hardclip_low > querypos) {
      /* startpos = hardclip_low; */
      starti = hardclip_low - querypos;
      hardclip_end_p = true;
      debug2(printf("  Setting starti %d = hardclip_low %d - querypos %d\n",
		    starti,hardclip_low,querypos));
    } else {
      /* startpos = querypos; */
      starti = 0;
    }

    if (querylength - hardclip_high < querypos + stringlength) {
      endpos = querylength - hardclip_high;
      endi = (querylength - hardclip_high) - querypos;
      debug2(printf("  Setting endi %d = (querylength %d - hardclip_high %d) - querypos %d\n",
		    endi,querylength,hardclip_high,querypos));
    } else {
      endpos = querypos + stringlength;
      endi = stringlength;
    }

    debug2(printf("  Counting matches from %d to %d\n",starti,endi));

    if (genomicfwd_refdiff == NULL) {
      if (endi > starti) {
	nmatches += (endi - starti);
      }

    } else if (md_lowercase_variant_p == false) {
      for (i = starti; i < endi; i++) {
	if (isupper(genomicfwd_refdiff[i])) {
	  if (nmismatches > 0 /*|| hardclip_end_p == true*/) {
	    FPRINTF(fp,"%dX",nmismatches);
	    nmismatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmatches++;

	} else {
	  /* A true mismatch against both variants */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
	}
      }

    } else {
      for (i = starti; i < endi; i++) {
	if (isupper(genomicfwd_refdiff[i])) {
	  if (nmismatches > 0 /*|| hardclip_end_p == true*/) {
	    FPRINTF(fp,"%dX",nmismatches);
	    nmismatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmatches++;

#if 0
	} else if (isupper(genomicfwd_bothdiff[i])) {
	  /* A mismatch against the reference only => alternate variant */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
#endif

	} else {
	  /* A true mismatch against both variants */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
	}
      }
    }

    if (nmatches > 0) {
      FPRINTF(fp,"%d=",nmatches);
      printp = true;
    } else if (nmismatches > 0) {
      FPRINTF(fp,"%dX",nmismatches);
      printp = true;
    }

    if (lastp == true) {
      /* cliplength = querypos + stringlength - endpos; */
      cliplength = querylength - endpos;
      if (cliplength > 0) {
	debug1(printf("  Pushing final %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
    }

  } else {
    debug2(printf("\nEntering print_extended_cigar with querypos %d, querylength %d, hardclip_low %d, hardclip_high %d, minus: %s ref\n",
		  querypos,querylength,hardclip_low,hardclip_high,genomicfwd_refdiff));
    querypos = querylength - querypos - stringlength;
    debug2(printf("  Revising querypos to be %d\n",querypos));

    if (hardclip_low == 0) {
      starti = 0;
      hardclip_end_p = true;
    } else if (hardclip_low > querypos) {
      /* startpos = hardclip_low; */
      starti = hardclip_low - querypos;
      hardclip_end_p = true;
      debug2(printf("  Setting starti %d = hardclip_low %d - querypos %d\n",
		    starti,hardclip_low,querypos));
    } else {
      /* startpos = querypos; */
      starti = 0;
    }

    if (querylength - hardclip_high < querypos + stringlength) {
      endpos = querylength - hardclip_high;
      endi = (querylength - hardclip_high) - querypos;
      debug2(printf("  Setting endi %d = (querylength %d - hardclip_high %d) - querypos %d\n",
		    endi,querylength,hardclip_high,querypos));
    } else {
      endpos = querypos + stringlength;
      endi = stringlength;
    }

    debug2(printf("  Counting matches from %d to %d\n",starti,endi));

    if (genomicfwd_refdiff == NULL) {
      if (endi > starti) {
	nmatches += (endi - starti);
      }

    } else if (md_lowercase_variant_p == false) {
      for (i = starti; i < endi; i++) {
	if (isupper(genomicfwd_refdiff[i])) {
	  if (nmismatches > 0 /*|| hardclip_end_p == true*/) {
	    FPRINTF(fp,"%dX",nmismatches);
	    nmismatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmatches++;

	} else {
	  /* A true mismatch against both variants */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
	}
      }

    } else {
      for (i = starti; i < endi; i++) {
	if (isupper(genomicfwd_refdiff[i])) {
	  if (nmismatches > 0 /*|| hardclip_end_p == true*/) {
	    FPRINTF(fp,"%dX",nmismatches);
	    nmismatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmatches++;

#if 0
	} else if (isupper(genomicfwd_bothdiff[i])) {
	  /* A mismatch against the reference only => alternate variant */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
#endif

	} else {
	  /* A true mismatch against both variants */
	  if (nmatches > 0 || hardclip_end_p == true) {
	    FPRINTF(fp,"%d=",nmatches);
	    nmatches = 0;
	    hardclip_end_p = false;
	    printp = true;
	  }
	  nmismatches++;
	}
      }
    }

    if (nmatches > 0) {
      FPRINTF(fp,"%d=",nmatches);
      printp = true;
    } else if (nmismatches > 0) {
      FPRINTF(fp,"%dX",nmismatches);
      printp = true;
    }

    if (lastp == true) {
      cliplength = endpos;
      if (cliplength > 0) {
	debug1(printf("  Pushing final %dH\n",cliplength));
	if (sam_hardclip_use_S_p) {
	  FPRINTF(fp,"%dS",cliplength);
	} else {
	  FPRINTF(fp,"%dH",cliplength);
	}
      }
    }
  }

  return printp;
}


static char complCode[128] = COMPLEMENT_LC;

static void
make_complement_buffered (char *complement, char *sequence, unsigned int length) {
  int i, j;

  /* complement = (char *) CALLOC(length+1,sizeof(char)); */
  for (i = length-1, j = 0; i >= 0; i--, j++) {
    complement[j] = complCode[(int) sequence[i]];
  }
  complement[length] = '\0';
  return;
}


/* Returns true if we printed an M, =, or X entry */
static bool
print_cigar_M (Filestring_T fp, Substring_T *substring_low, Substring_T substring,
	       int substring_length, int substring_start,
	       int stringlength, int querypos, int querylength,
	       int hardclip_low, int hardclip_high, bool plusp, bool lastp, int trimlength) {
  bool printp;
  char *genomicfwd_refdiff, *genomicdir_refdiff;
  
  if (cigar_extended_p == false) {
    printp = print_cigar(fp,/*type*/'M',stringlength,querypos,querylength,
			 hardclip_low,hardclip_high,plusp,lastp,trimlength);

  } else if ((genomicdir_refdiff = Substring_genomic_refdiff(substring)) == NULL) {
    printp = print_extended_cigar(fp,/*genomicfwd_refdiff*/NULL,/*stringlength*/substring_length,
				  /*querypos*/substring_start,querylength,
				  hardclip_low,hardclip_high,plusp,lastp);
  } else if (plusp == true) {
    printp = print_extended_cigar(fp,&(genomicdir_refdiff[substring_start]),/*stringlength*/substring_length,
				  /*querypos*/substring_start,querylength,
				  hardclip_low,hardclip_high,plusp,lastp);
  } else {
    genomicfwd_refdiff = (char *) MALLOCA((substring_length+1) * sizeof(char));
    make_complement_buffered(genomicfwd_refdiff,&(genomicdir_refdiff[substring_start]),substring_length);
    printp = print_extended_cigar(fp,genomicfwd_refdiff,/*stringlength*/substring_length,
				  /*querypos*/substring_start,querylength,
				  hardclip_low,hardclip_high,plusp,lastp);
    FREEA(genomicfwd_refdiff);
  }

  if (*substring_low == NULL && printp == true) {
    *substring_low = substring;
  }

  return printp;
}


#if 0
/* Copy also in pair.c for GMAP */
static bool
check_cigar_types (Intlist_T cigar_types) {
  Intlist_T p;
  int type;
  bool M_present_p = false;

  for (p = cigar_types; p != NULL; p = Intlist_next(p)) {
    type = Intlist_head(p);
    if (type == 'M') {
      M_present_p = true;
#if 0
    } else if (type == 'H' && last_type == 'S') {
      debug1(printf("check_cigar_types detects adjacent S and H, so returning false\n"));
      return false;
    } else if (type == 'S' && last_type == 'H') {
      debug1(printf("check_cigar_types detects adjacent S and H, so returning false\n"));
      return false;
#endif
    }
  }

  return M_present_p;
}
#endif



/* Modified from print_substrings */
int
Cigar_length_substrings (Stage3end_T stage3end, int querylength, int hardclip_low, int hardclip_high) {
  int length = 0;
  List_T startp, startq, endp;
  /* List_T prevp, nextp, finalp; */

  Substring_T substring, substringL, substringH;
  /* Junction_T post_junction; */
  /* int type; */

  List_T substrings_LtoH, junctions_LtoH;
  List_T p, q;
  /* int substring_start, substring_length; */

  bool plusp;
#ifdef PRINT_AMBIG_COORDS
  Univcoord_T chroffset;
#endif

  
  if ((substrings_LtoH = Stage3end_substrings_LtoH(stage3end)) == NULL) {
    return 0;
  } else {
    plusp = Stage3end_plusp(stage3end);
    substrings_LtoH = Stage3end_substrings_LtoH(stage3end);
    junctions_LtoH = Stage3end_junctions_LtoH(stage3end);
    substringL = (Substring_T) List_head(substrings_LtoH);
    substringH = (Substring_T) List_last_value(substrings_LtoH,NULL);
  }

  if (Substring_has_alts_p(substringL) == true) {
    /* prevp = substrings_LtoH; */
    startp = List_next(substrings_LtoH);
    startq = List_next(junctions_LtoH);
  } else {
    /* prevp = (List_T) NULL; */
    startp = substrings_LtoH;
    startq = junctions_LtoH;
  }
  if (Substring_has_alts_p(substringH) == true) {
    endp = List_last_item(substrings_LtoH,NULL);
  } else {
    endp = (List_T) NULL;
  }

  debug(printf("End has %d substrings\n",List_length(substrings_LtoH)));

  p = startp;
  q = startq;

  if (plusp == true) {
    /* Plus */
    while (p != endp && Substring_queryend((Substring_T) List_head(p)) < hardclip_low) {
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      /* prevp = p; */
      p = List_next(p);
      q = List_next(q);
    }

    if (p == NULL) {
      debug(printf("Empty substring\n"));

    } else {
      substring = (Substring_T) List_head(p);
      if (List_next(p) == endp || Substring_queryend(substring) >= querylength - hardclip_high) {
	/* Single substring */
	debug(printf("Single substring %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));
	
	if (hide_soft_clips_p == true) {
	  /* substring_start = Substring_querystart_pretrim(substring); */
	  /* substring_length = Substring_match_length_pretrim(substring); */
	  length += length_cigar_M(Substring_querystart(substring) + Substring_match_length(substring) +
				   (querylength - Substring_queryend(substring)),/*querypos*/0,querylength,
				   hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true);
	} else {
	  /* substring_start = Substring_querystart(substring); */
	  /* substring_length = Substring_match_length(substring); */
#if 0
	  print_cigar(fp,/*type*/'S',Substring_querystart(substring),
		      /*querypos*/0,querylength,hardclip_low,hardclip_high,
		      /*plusp*/true,/*lastp*/false,/*trimlength*/0);
#endif
	  length += length_cigar_M(Substring_match_length(substring),
				   /*querypos*/Substring_querystart(substring),querylength,
				   hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false);
#if 0
	  print_cigar(fp,/*type*/'S',querylength - Substring_queryend(substring),
		      /*querypos*/Substring_queryend(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
#endif
	}
	/* finalp = p; */
	/* nextp = List_next(p); */
	
      } else {
	/* First substring, plus */
	debug(printf("First substring, plus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));
	
	/* post_junction = (Junction_T) List_head(q); */
	
	if (hide_soft_clips_p == true) {
	  /* substring_start = Substring_querystart_pretrim(substring); */
	  /* substring_length = Substring_match_length_pretrim(substring); */
	  length += length_cigar_M(Substring_querystart(substring) +
				   Substring_match_length(substring),
				   /*querypos*/0,querylength,hardclip_low,hardclip_high,
				   /*plusp*/true,/*lastp*/false);
	} else {
	  /* substring_start = Substring_querystart(substring); */
	  /* substring_length = Substring_match_length(substring); */
#if 0
	  print_cigar(fp,/*type*/'S',Substring_querystart(substring),
		      /*querypos*/0,querylength,hardclip_low,hardclip_high,
		      /*plusp*/true,/*lastp*/false,/*trimlength*/0);
#endif
	  length += length_cigar_M(Substring_match_length(substring),
				   /*querypos*/Substring_querystart(substring),querylength,
				   hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false);
	}
	p = List_next(p);
      
	while (p != endp && Substring_queryend((Substring_T) List_head(p)) < querylength - hardclip_high) {
#if 0
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    debug1(printf("1. Pushing %dD\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dD",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == INS_JUNCTION) {
	    debug1(printf("1. Pushing %dI\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dI",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == SPLICE_JUNCTION) {
	    debug1(printf("1. Pushing %dN\n",Junction_splice_distance(post_junction)));
	    FPRINTF(fp,"%uN",Junction_splice_distance(post_junction));
	  }
#endif
	  q = List_next(q);
#if 0
	  if (q == NULL) {
	  } else {
	    post_junction = (Junction_T) List_head(q);
	  }
#endif
	  
	  substring = (Substring_T) List_head(p);
	  if (List_next(p) == endp) {
	    /* Last substring, plus, not hard-clipped */
	    debug(printf("Last substring, plus, not hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
			 Substring_queryend((Substring_T) List_head(p))));
	    
	    if (hide_soft_clips_p == true) {
	      /* substring_start = Substring_querystart_pretrim(substring); */
	      /* substring_length = Substring_match_length_pretrim(substring); */
	      length += length_cigar_M(Substring_match_length(substring) +
				       (querylength - Substring_queryend(substring)),
				       /*querypos*/Substring_querystart(substring),querylength,
				       hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true);
	    } else {
	      /* substring_start = Substring_querystart(substring); */
	      /* substring_length = Substring_match_length(substring); */
	      length += length_cigar_M(Substring_match_length(substring),
				       /*querypos*/Substring_querystart(substring),querylength,
				       hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false);
#if 0
	      print_cigar(fp,/*type*/'S',querylength - Substring_queryend(substring),
			  /*querypos*/Substring_queryend(substring),querylength,
			  hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
#endif
	    }
	    /* finalp = p; */
	    /* nextp = List_next(p); */
	    
	  } else {
	    /* Middle substring, plus */
	    debug(printf("Middle substring, plus %d..%d\n",Substring_querystart((Substring_T) List_head(p)), 
			 Substring_queryend((Substring_T) List_head(p))));
	    /* substring_start = Substring_querystart(substring); */
	    /* substring_length = Substring_match_length(substring); */
	    
	    length += length_cigar_M(Substring_match_length(substring),
				     /*querypos*/Substring_querystart(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false);
	  }
	  p = List_next(p);
	}
      
	if (p != endp) {
#if 0
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    debug1(printf("2. Pushing %dD\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dD",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == INS_JUNCTION) {
	    debug1(printf("2. Pushing %dI\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dI",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == SPLICE_JUNCTION) {
	    debug1(printf("2. Pushing %dN\n",Junction_splice_distance(post_junction)));
	    FPRINTF(fp,"%uN",Junction_splice_distance(post_junction));
	  }
#endif
	  
	  /* Last substring, plus, hard-clipped */
	  substring = (Substring_T) List_head(p);
	  debug(printf("Last substring, plus, hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		       Substring_queryend((Substring_T) List_head(p))));
	  if (hide_soft_clips_p == true) {
	    /* substring_start = Substring_querystart_pretrim(substring); */
	    /* substring_length = Substring_match_length_pretrim(substring); */
	    length += length_cigar_M(Substring_match_length(substring) +
				     (querylength - Substring_queryend(substring)),
				     /*querypos*/Substring_querystart(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true);
	  } else {
	    /* substring_start = Substring_querystart(substring); */
	    /* substring_length = Substring_match_length(substring); */
	    length += length_cigar_M(Substring_match_length(substring),
				     /*querypos*/Substring_querystart(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false);
#if 0
	    print_cigar(fp,/*type*/'S',querylength - Substring_queryend(substring),
			/*querypos*/Substring_queryend(substring),querylength,
			hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
#endif
	  }
	  /* finalp = p; */
	  /* nextp = List_next(p); */
	  
	}
      }
    }

  } else {
    /* Minus */
    while (p != endp && Substring_querystart((Substring_T) List_head(p)) >= querylength - hardclip_low) {
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      /* prevp = p; */
      p = List_next(p);
      q = List_next(q);
    }

    if (p == NULL) {
      debug(printf("Empty substring\n"));

    } else {
      substring = (Substring_T) List_head(p);
      if (List_next(p) == endp || Substring_querystart(substring) < hardclip_high) {
	/* Single substring */
	debug(printf("Single substring %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));
	
	if (hide_soft_clips_p == true) {
	  /* substring_start = Substring_querystart_pretrim(substring); */
	  /* substring_length = Substring_match_length_pretrim(substring); */
	  length += length_cigar_M((querylength - Substring_queryend(substring)) + 
				   Substring_match_length(substring) + Substring_querystart(substring),
				   /*querypos*/querylength,querylength,hardclip_low,hardclip_high,
				   /*plusp*/false,/*lastp*/true);
	} else {
	  /* substring_start = Substring_querystart(substring); */
	  /* substring_length = Substring_match_length(substring); */
#if 0
	  print_cigar(fp,/*type*/'S',querylength - Substring_queryend(substring),
		      /*querypos*/querylength,querylength,
		      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
#endif
	  length += length_cigar_M(Substring_match_length(substring),
				   /*querypos*/Substring_queryend(substring),querylength,
				   hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false);
#if 0
	  print_cigar(fp,/*type*/'S',Substring_querystart(substring),
		      /*querypos*/Substring_querystart(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true,/*trimlength*/0);
#endif
	}
	/* finalp = p; */
	/* nextp = List_next(p); */
	
      } else {
	/* First substring, minus */
	debug(printf("First substring, minus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));
	
	/* post_junction = (Junction_T) List_head(q); */
	
	if (hide_soft_clips_p == true) {
	  /* substring_start = Substring_querystart_pretrim(substring); */
	  /* substring_length = Substring_match_length_pretrim(substring); */
	  length += length_cigar_M((querylength - Substring_queryend(substring)) +
				   Substring_match_length(substring),
				   /*querypos*/querylength,querylength,hardclip_low,hardclip_high,
				   /*plusp*/false,/*lastp*/false);
	} else {
	  /* substring_start = Substring_querystart(substring); */
	  /* substring_length = Substring_match_length(substring); */
#if 0
	  print_cigar(fp,/*type*/'S',querylength - Substring_queryend(substring),
		      /*querypos*/querylength,querylength,
		      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
#endif
	  length += length_cigar_M(Substring_match_length(substring),
				   /*querypos*/Substring_queryend(substring),querylength,
				   hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false);
	}
	p = List_next(p);
	
	while (p != endp && Substring_querystart((Substring_T) List_head(p)) >= hardclip_high) {
#if 0
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    debug1(printf("3. Pushing %dD\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dD",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == INS_JUNCTION) {
	    debug1(printf("3. Pushing %dI\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dI",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == SPLICE_JUNCTION) {
	    debug1(printf("3. Pushing %dN\n",Junction_splice_distance(post_junction)));
	    FPRINTF(fp,"%uN",Junction_splice_distance(post_junction));
	  }
#endif
	  q = List_next(q);
#if 0
	  if (q == NULL) {
	  } else {
	    post_junction = (Junction_T) List_head(q);
	  }
#endif
	  
	  substring = (Substring_T) List_head(p);
	  if (List_next(p) == endp) {
	    /* Last substring, minus, not hard-clipped */
	    debug(printf("Last substring, minus, not hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
			 Substring_queryend((Substring_T) List_head(p))));
	    
	    if (hide_soft_clips_p == true) {
	      /* substring_start = Substring_querystart_pretrim(substring); */
	      /* substring_length = Substring_match_length_pretrim(substring); */
	      length += length_cigar_M(Substring_match_length(substring) +
				       Substring_querystart(substring),
				       /*querypos*/Substring_queryend(substring),querylength,
				       hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true);
	    } else {
	      /* substring_start = Substring_querystart(substring); */
	      /* substring_length = Substring_match_length(substring); */
	      length += length_cigar_M(Substring_match_length(substring),
				       /*querypos*/Substring_queryend(substring),querylength,
				       hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false);
#if 0
	      print_cigar(fp,/*type*/'S',Substring_querystart(substring),
			  /*querypos*/Substring_querystart(substring),querylength,hardclip_low,hardclip_high,
			  /*plusp*/false,/*lastp*/true,/*trimlength*/0);
#endif
	    }
	    /* finalp = p; */
	    /* nextp = List_next(p); */
	    
	  } else {
	    /* Middle substring, minus */
	    debug(printf("Middle substring, minus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
			 Substring_queryend((Substring_T) List_head(p))));
	    /* substring_start = Substring_querystart(substring); */
	    /* substring_length = Substring_match_length(substring); */
	    
	    length += length_cigar_M(Substring_match_length(substring),
				     /*querypos*/Substring_queryend(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false);
	  }
	  p = List_next(p);
	}
	
	if (p != endp) {
#if 0
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    debug1(printf("4. Pushing %dD\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dD",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == INS_JUNCTION) {
	    debug1(printf("4. Pushing %dI\n",Junction_nindels(post_junction)));
	    FPRINTF(fp,"%dI",Junction_nindels(post_junction));
	    nindels += Junction_nindels(post_junction);
	  } else if (type == SPLICE_JUNCTION) {
	    debug1(printf("4. Pushing %dN\n",Junction_splice_distance(post_junction)));
	    FPRINTF(fp,"%uN",Junction_splice_distance(post_junction));
	  }
#endif
	  
	  /* Last substring, minus, hard-clipped */
	  substring = (Substring_T) List_head(p);
	  debug(printf("Last substring, minus, hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		       Substring_queryend((Substring_T) List_head(p))));
	  
	  if (hide_soft_clips_p == true) {
	    /* substring_start = Substring_querystart_pretrim(substring); */
	    /* substring_length = Substring_match_length_pretrim(substring); */
	    length += length_cigar_M(Substring_querystart(substring),
				     /*querypos*/Substring_queryend(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true);
	  } else {
	    /* substring_start = Substring_querystart(substring); */
	    /* substring_length = Substring_match_length(substring); */
	    length += length_cigar_M(Substring_match_length(substring),
				     /*querypos*/Substring_queryend(substring),querylength,
				     hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false);
#if 0
	    print_cigar(fp,/*type*/'S',Substring_querystart(substring),
			/*querypos*/Substring_querystart(substring),querylength,hardclip_low,hardclip_high,
			/*plusp*/false,/*lastp*/true,/*trimlength*/0);
#endif
	  }
	  /* finalp = p; */
	  /* nextp = List_next(p); */
	  
	}
      }
    }
  }

  return length;
}


static void
print_substrings (int *nindels, int *nmismatches_refdiff, int *nmismatches_bothdiff,
		  Substring_T *substring_low, List_T *prevp, List_T *nextp, List_T *finalp,
		  Filestring_T cigar_fp, Filestring_T md_fp,
		  List_T substrings_LtoH, List_T junctions_LtoH, bool plusp,
		  int querylength, int hardclip_low, int hardclip_high) {
  Substring_T substring, substringL, substringH;
  List_T startp, startq, endp;

  Junction_T post_junction;
  int type;

  List_T p, q;
  int substring_start, substring_length;

  bool M_printedp = false;

  int matchlength = 0;
  bool md_printp = false, md_lastp;
  char *deletion_string;

#ifdef PRINT_AMBIG_COORDS
  Univcoord_T chroffset;
#endif

  
  debug(printf("\n\n***Entered print_substrings with plusp %d, hardclip_low %d and hardclip_high %d\n",
	       plusp,hardclip_low,hardclip_high));
  *substring_low = (Substring_T) NULL;

  if (substrings_LtoH == NULL) {
    FPRINTF(cigar_fp,"*");
    return;

  } else {
#ifdef DEBUG
    for (p = substrings_LtoH; p != NULL; p = List_next(p)) {
      substring = (Substring_T) List_head(p);
      printf("%d..%d #%d:%u..%u\n",
	     Substring_querystart(substring),Substring_queryend(substring),
	     Substring_chrnum(substring),Substring_alignstart_trim_chr(substring),
	     Substring_alignend_trim_chr(substring));
    }
    printf("\n");
#endif

    substringL = (Substring_T) List_head(substrings_LtoH);
    substringH = (Substring_T) List_last_value(substrings_LtoH,NULL);
  }


  if (Substring_has_alts_p(substringL) == true) {
    *prevp = substrings_LtoH;
    startp = List_next(substrings_LtoH);
    startq = List_next(junctions_LtoH);
  } else {
    *prevp = (List_T) NULL;
    startp = substrings_LtoH;
    startq = junctions_LtoH;
  }
  if (Substring_has_alts_p(substringH) == true) {
    *nextp = endp = List_last_item(substrings_LtoH,NULL);
  } else {
    *nextp = endp = (List_T) NULL;
  }

  debug(printf("End has %d substrings\n",List_length(substrings_LtoH)));


  /* Revise startp and endp for hardclips */
  if (plusp == true) {
    /* Plus */
    p = substrings_LtoH;
    q = junctions_LtoH;
    if (p != startp) {
      p = List_next(p);
      q = List_next(q);
    }
    assert(p == startp);
    assert(q == startq);
    
    while (p != NULL && Substring_queryend((Substring_T) List_head(p)) <= hardclip_low) { /* was <, but that allowed a gap before the hardclip */
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d at start (plus)\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      p = List_next(p);
      q = List_next(q);
      *prevp = (List_T) NULL;
    }
    startp = p;
    startq = q;

    p = substrings_LtoH = List_reverse(substrings_LtoH);
    if (endp != NULL) {
      p = List_next(p);
    }

    while (p != NULL && Substring_querystart((Substring_T) List_head(p)) >= querylength - hardclip_high) {
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d at end (plus)\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      endp = p;
      p = List_next(p);
      *nextp = (List_T) NULL;
    }

    substrings_LtoH = List_reverse(substrings_LtoH);

  } else {
    /* Minus */
    p = substrings_LtoH;
    q = junctions_LtoH;
    if (p != startp) {
      p = List_next(p);
      q = List_next(q);
    }
    assert(p == startp);
    assert(q == startq);

    while (p != NULL && Substring_querystart((Substring_T) List_head(p)) >= querylength - hardclip_low) {
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d at start (minus)\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      p = List_next(p);
      q = List_next(q);
      *prevp = (List_T) NULL;
    }
    startp = p;
    startq = q;

    p = substrings_LtoH = List_reverse(substrings_LtoH);
    if (endp != NULL) {
      p = List_next(p);
    }

    while (p != NULL && Substring_queryend((Substring_T) List_head(p)) <= hardclip_high) { /* was <, but that allowed a gap before the hardclip */
      /* Skip, because substring entirely in hard-clipped region */
      debug(printf("Skipping %d..%d at end (minus)\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
      endp = p;
      p = List_next(p);
      *nextp = (List_T) NULL;
    }

    substrings_LtoH = List_reverse(substrings_LtoH);
  }


  p = startp;
  q = startq;

  if (p == NULL) {
    debug(printf("Empty substring\n"));
    FPRINTF(cigar_fp,"*");

  } else if (plusp == true) {
    /* Plus */
    substring = (Substring_T) List_head(p);
    if (startp == endp || List_next(p) == endp) {
      /* Single substring */
      debug(printf("Single substring %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
	
      if (hide_soft_clips_p == true) {
	substring_start = Substring_querystart_pretrim(substring);
	substring_length = Substring_match_length_pretrim(substring);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_querystart(substring) + Substring_match_length(substring) +
				    (querylength - Substring_queryend(substring)),/*querypos*/0,querylength,
				    hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);

      } else {
	substring_start = Substring_querystart(substring);
	substring_length = Substring_match_length(substring);
	print_cigar(cigar_fp,/*type*/'S',Substring_querystart(substring),
		    /*querypos*/0,querylength,hardclip_low,hardclip_high,
		    /*plusp*/true,/*lastp*/false,/*trimlength*/0);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_match_length(substring),
				    /*querypos*/Substring_querystart(substring),querylength,
				    hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);

	print_cigar(cigar_fp,/*type*/'S',querylength - Substring_queryend(substring),
		    /*querypos*/Substring_queryend(substring),querylength,
		    hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
      }

      matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
				       matchlength,md_fp,substring,substring_length,substring_start,
				       querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
				       /*plusp*/true,/*lastp*/true);

      *finalp = p;
	
    } else {
      /* First substring, plus */
      debug(printf("First substring, plus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
	
      if (hide_soft_clips_p == true) {
	substring_start = Substring_querystart_pretrim(substring);
	substring_length = Substring_match_length_pretrim(substring);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_querystart(substring) +
				    Substring_match_length(substring),
				    /*querypos*/0,querylength,hardclip_low,hardclip_high,
				    /*plusp*/true,/*lastp*/false,/*trimlength*/0);

      } else {
	substring_start = Substring_querystart(substring);
	substring_length = Substring_match_length(substring);
	print_cigar(cigar_fp,/*type*/'S',Substring_querystart(substring),
		    /*querypos*/0,querylength,hardclip_low,hardclip_high,
		    /*plusp*/true,/*lastp*/false,/*trimlength*/0);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_match_length(substring),
				    /*querypos*/Substring_querystart(substring),querylength,
				    hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);
      }

      post_junction = (Junction_T) List_head(q);
      if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	md_lastp = true;
      } else {
	md_lastp = false;
      }
      matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
				       matchlength,md_fp,substring,substring_length,substring_start,
				       querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
				       /*plusp*/true,md_lastp);

      p = List_next(p);
	
      while (p != endp) {
	if (M_printedp == false) {
	  debug1(printf("Skipping initial indel or splice\n"));

	} else if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	  debug1(printf("1. Pushing %dD\n",Junction_nindels(post_junction)));
	  FPRINTF(cigar_fp,"%dD",Junction_nindels(post_junction));
	  *nindels += Junction_nindels(post_junction);

	  deletion_string = Junction_deletion_string(post_junction,/*plusp*/true); 
	  FPRINTF(md_fp,"^%s",deletion_string);
	  FREE(deletion_string);

	} else if (type == INS_JUNCTION) {
	  debug1(printf("1. Pushing %dI\n",Junction_nindels(post_junction)));
#if 0
	  FPRINTF(cigar_fp,"%dI",Junction_nindels(post_junction));
#else
	  print_cigar(cigar_fp,/*type*/'I',/*stringlength*/Junction_nindels(post_junction),
		      /*querypos*/Substring_queryend(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);
#endif
	  *nindels += Junction_nindels(post_junction);

	} else if (type == SPLICE_JUNCTION) {
	  debug1(printf("1. Pushing %dN\n",Junction_splice_distance(post_junction)));
	  FPRINTF(cigar_fp,"%uN",Junction_splice_distance(post_junction));
	}

	q = List_next(q);
	if (q == NULL) {
	  md_lastp = true;
	} else {
	  post_junction = (Junction_T) List_head(q);
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    md_lastp = true;
	  } else {
	    md_lastp = false;
	  }
	}
	  
	substring = (Substring_T) List_head(p);
	if (List_next(p) == endp) {
	  /* Last substring, plus, not hard-clipped */
	  debug(printf("Last substring, plus, not hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		       Substring_queryend((Substring_T) List_head(p))));
	    
	  if (hide_soft_clips_p == true) {
	    substring_start = Substring_querystart_pretrim(substring);
	    substring_length = Substring_match_length_pretrim(substring);
	    M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
					substring_length,substring_start,
					Substring_match_length(substring) +
					(querylength - Substring_queryend(substring)),
					/*querypos*/Substring_querystart(substring),querylength,
					hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);

	  } else {
	    substring_start = Substring_querystart(substring);
	    substring_length = Substring_match_length(substring);
	    M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
					substring_length,substring_start,Substring_match_length(substring),
					/*querypos*/Substring_querystart(substring),querylength,
					hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);

	    print_cigar(cigar_fp,/*type*/'S',querylength - Substring_queryend(substring),
			/*querypos*/Substring_queryend(substring),querylength,
			hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
	  }

	  matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					   matchlength,md_fp,substring,substring_length,substring_start,
					   querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					   /*plusp*/true,/*lastp*/true);

	  *finalp = p;
	    
	} else {
	  /* Middle substring, plus */
	  debug(printf("Middle substring, plus %d..%d\n",Substring_querystart((Substring_T) List_head(p)), 
		       Substring_queryend((Substring_T) List_head(p))));
	  substring_start = Substring_querystart(substring);
	  substring_length = Substring_match_length(substring);
	    
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring),
				      /*querypos*/Substring_querystart(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);

	  matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					   matchlength,md_fp,substring,substring_length,substring_start,
					   querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					   /*plusp*/true,md_lastp);
	}
	p = List_next(p);
      }
	
      if (p != endp) {
	if (M_printedp == false) {
	  debug1(printf("Skipping initial indel or splice\n"));

	} else if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	  debug1(printf("2. Pushing %dD\n",Junction_nindels(post_junction)));
	  FPRINTF(cigar_fp,"%dD",Junction_nindels(post_junction));
	  *nindels += Junction_nindels(post_junction);

	  deletion_string = Junction_deletion_string(post_junction,/*plusp*/true); 
	  FPRINTF(md_fp,"^%s",deletion_string);
	  FREE(deletion_string);

	} else if (type == INS_JUNCTION) {
	  debug1(printf("2. Pushing %dI\n",Junction_nindels(post_junction)));
#if 0
	  FPRINTF(cigar_fp,"%dI",Junction_nindels(post_junction));
#else
	  print_cigar(cigar_fp,/*type*/'I',/*stringlength*/Junction_nindels(post_junction),
		      /*querypos*/Substring_queryend(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);
#endif
	  *nindels += Junction_nindels(post_junction);

	} else if (type == SPLICE_JUNCTION) {
	  debug1(printf("2. Pushing %dN\n",Junction_splice_distance(post_junction)));
	  FPRINTF(cigar_fp,"%uN",Junction_splice_distance(post_junction));
	}
	  
	/* Last substring, plus, hard-clipped */
	substring = (Substring_T) List_head(p);
	debug(printf("Last substring, plus, hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));

	if (hide_soft_clips_p == true) {
	  substring_start = Substring_querystart_pretrim(substring);
	  substring_length = Substring_match_length_pretrim(substring);
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring) +
				      (querylength - Substring_queryend(substring)),
				      /*querypos*/Substring_querystart(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);

	} else {
	  substring_start = Substring_querystart(substring);
	  substring_length = Substring_match_length(substring);
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring),
				      /*querypos*/Substring_querystart(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/false,/*trimlength*/0);

	  print_cigar(cigar_fp,/*type*/'S',querylength - Substring_queryend(substring),
		      /*querypos*/Substring_queryend(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/true,/*lastp*/true,/*trimlength*/0);
	}

	matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					 matchlength,md_fp,substring,substring_length,substring_start,
					 querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					 /*plusp*/true,/*lastp*/true);

	*finalp = p;
      }
    }

  } else {
    /* Minus */
    substring = (Substring_T) List_head(p);
    if (startp == endp || List_next(p) == endp) {
      /* Single substring */
      debug(printf("Single substring %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
	
      if (hide_soft_clips_p == true) {
	substring_start = Substring_querystart_pretrim(substring);
	substring_length = Substring_match_length_pretrim(substring);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    (querylength - Substring_queryend(substring)) + 
				    Substring_match_length(substring) + Substring_querystart(substring),
				    /*querypos*/querylength,querylength,hardclip_low,hardclip_high,
				    /*plusp*/false,/*lastp*/true,/*trimlength*/0);

      } else {
	substring_start = Substring_querystart(substring);
	substring_length = Substring_match_length(substring);
	print_cigar(cigar_fp,/*type*/'S',querylength - Substring_queryend(substring),
		    /*querypos*/querylength,querylength,
		    hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_match_length(substring),
				    /*querypos*/Substring_queryend(substring),querylength,
				    hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);

	print_cigar(cigar_fp,/*type*/'S',Substring_querystart(substring),
		    /*querypos*/Substring_querystart(substring),querylength,
		    hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true,/*trimlength*/0);
      }

      matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
				       matchlength,md_fp,substring,substring_length,substring_start,
				       querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
				       /*plusp*/false,/*lastp*/true);

      *finalp = p;
	
    } else {
      /* First substring, minus */
      debug(printf("First substring, minus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		   Substring_queryend((Substring_T) List_head(p))));
	
      if (hide_soft_clips_p == true) {
	substring_start = Substring_querystart_pretrim(substring);
	substring_length = Substring_match_length_pretrim(substring);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    (querylength - Substring_queryend(substring)) +
				    Substring_match_length(substring),
				    /*querypos*/querylength,querylength,hardclip_low,hardclip_high,
				    /*plusp*/false,/*lastp*/false,/*trimlength*/0);

      } else {
	substring_start = Substring_querystart(substring);
	substring_length = Substring_match_length(substring);
	print_cigar(cigar_fp,/*type*/'S',querylength - Substring_queryend(substring),
		    /*querypos*/querylength,querylength,
		    hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
	M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				    substring_length,substring_start,
				    Substring_match_length(substring),
				    /*querypos*/Substring_queryend(substring),querylength,
				    hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
      }

      post_junction = (Junction_T) List_head(q);
      if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	md_lastp = true;
      } else {
	md_lastp = false;
      }
      matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
				       matchlength,md_fp,substring,substring_length,substring_start,
				       querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
				       /*plusp*/false,md_lastp);

      p = List_next(p);
	
      while (p != endp) {
	if (M_printedp == false) {
	  debug1(printf("Skipping initial indel or splice\n"));

	} else if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	  debug1(printf("3. Pushing %dD\n",Junction_nindels(post_junction)));
	  FPRINTF(cigar_fp,"%dD",Junction_nindels(post_junction));
	  *nindels += Junction_nindels(post_junction);

	  deletion_string = Junction_deletion_string(post_junction,/*plusp:true*/true); 
	  FPRINTF(md_fp,"^%s",deletion_string);
	  FREE(deletion_string);

	} else if (type == INS_JUNCTION) {
	  debug1(printf("3. Pushing %dI\n",Junction_nindels(post_junction)));
#if 0
	  FPRINTF(cigar_fp,"%dI",Junction_nindels(post_junction));
#else
	  print_cigar(cigar_fp,/*type*/'I',/*stringlength*/Junction_nindels(post_junction),
		      /*querypos*/Substring_querystart(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
#endif
	  *nindels += Junction_nindels(post_junction);

	} else if (type == SPLICE_JUNCTION) {
	  debug1(printf("3. Pushing %dN\n",Junction_splice_distance(post_junction)));
	  FPRINTF(cigar_fp,"%uN",Junction_splice_distance(post_junction));
	}
	q = List_next(q);
	if (q == NULL) {
	  md_lastp = true;
	} else {
	  post_junction = (Junction_T) List_head(q);
	  if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	    md_lastp = true;
	  } else {
	    md_lastp = false;
	  }
	}
	  
	substring = (Substring_T) List_head(p);
	if (List_next(p) == endp) {
	  /* Last substring, minus, not hard-clipped */
	  debug(printf("Last substring, minus, not hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		       Substring_queryend((Substring_T) List_head(p))));
	    
	  if (hide_soft_clips_p == true) {
	    substring_start = Substring_querystart_pretrim(substring);
	    substring_length = Substring_match_length_pretrim(substring);
	    M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
					substring_length,substring_start,
					Substring_match_length(substring) +
					Substring_querystart(substring),
					/*querypos*/Substring_queryend(substring),querylength,
					hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true,/*trimlength*/0);

	  } else {
	    substring_start = Substring_querystart(substring);
	    substring_length = Substring_match_length(substring);
	    M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
					substring_length,substring_start,
					Substring_match_length(substring),
					/*querypos*/Substring_queryend(substring),querylength,
					hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);

	    print_cigar(cigar_fp,/*type*/'S',Substring_querystart(substring),
			/*querypos*/Substring_querystart(substring),querylength,hardclip_low,hardclip_high,
			/*plusp*/false,/*lastp*/true,/*trimlength*/0);
	  }

	  matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					   matchlength,md_fp,substring,substring_length,substring_start,
					   querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					   /*plusp*/false,/*lastp*/true);

	  *finalp = p;
	    
	} else {
	  /* Middle substring, minus */
	  debug(printf("Middle substring, minus %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		       Substring_queryend((Substring_T) List_head(p))));
	  substring_start = Substring_querystart(substring);
	  substring_length = Substring_match_length(substring);
	    
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring),
				      /*querypos*/Substring_queryend(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);

	  matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					   matchlength,md_fp,substring,substring_length,substring_start,
					   querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					   /*plusp*/false,md_lastp);
	}
	p = List_next(p);
      }
	
      if (p != endp) {
	if (M_printedp == false) {
	  debug1(printf("Skipping initial indel or splice\n"));

	} else if ((type = Junction_type(post_junction)) == DEL_JUNCTION) {
	  debug1(printf("4. Pushing %dD\n",Junction_nindels(post_junction)));
	  FPRINTF(cigar_fp,"%dD",Junction_nindels(post_junction));
	  *nindels += Junction_nindels(post_junction);

	  deletion_string = Junction_deletion_string(post_junction,/*plusp:true*/true); 
	  FPRINTF(md_fp,"^%s",deletion_string);
	  FREE(deletion_string);

	} else if (type == INS_JUNCTION) {
	  debug1(printf("4. Pushing %dI\n",Junction_nindels(post_junction)));
#if 0
	  FPRINTF(cigar_fp,"%dI",Junction_nindels(post_junction));
#else
	  print_cigar(cigar_fp,/*type*/'I',/*stringlength*/Junction_nindels(post_junction),
		      /*querypos*/Substring_querystart(substring),querylength,
		      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);
#endif
	  *nindels += Junction_nindels(post_junction);

	} else if (type == SPLICE_JUNCTION) {
	  debug1(printf("4. Pushing %dN\n",Junction_splice_distance(post_junction)));
	  FPRINTF(cigar_fp,"%uN",Junction_splice_distance(post_junction));
	}
	  
	/* Last substring, minus, hard-clipped */
	substring = (Substring_T) List_head(p);
	debug(printf("Last substring, minus, hard-clipped %d..%d\n",Substring_querystart((Substring_T) List_head(p)),
		     Substring_queryend((Substring_T) List_head(p))));
	  
	if (hide_soft_clips_p == true) {
	  substring_start = Substring_querystart_pretrim(substring);
	  substring_length = Substring_match_length_pretrim(substring);
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring) +
				      Substring_querystart(substring),
				      /*querypos*/Substring_queryend(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/true,/*trimlength*/0);

	} else {
	  substring_start = Substring_querystart(substring);
	  substring_length = Substring_match_length(substring);
	  M_printedp |= print_cigar_M(cigar_fp,&(*substring_low),substring,
				      substring_length,substring_start,
				      Substring_match_length(substring),
				      /*querypos*/Substring_queryend(substring),querylength,
				      hardclip_low,hardclip_high,/*plusp*/false,/*lastp*/false,/*trimlength*/0);

	  print_cigar(cigar_fp,/*type*/'S',Substring_querystart(substring),
		      /*querypos*/Substring_querystart(substring),querylength,hardclip_low,hardclip_high,
		      /*plusp*/false,/*lastp*/true,/*trimlength*/0);
	}

	matchlength = MD_print_substring(&md_printp,&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
					 matchlength,md_fp,substring,substring_length,substring_start,
					 querylength,hardclip_low,hardclip_high,md_lowercase_variant_p,
					 /*plusp*/false,/*lastp*/true);

	*finalp = p;
      }
    }
  }

  if (md_printp == false) {
    FPRINTF(md_fp,"0");
  }

  return;
}


void
Cigar_compute_main (Filestring_T *cigar_fp, Filestring_T *md_fp,
		    int *nmismatches_refdiff, int *nmismatches_bothdiff,
		    int *sequence_hardclip_low, int *sequence_hardclip_high,
		    List_T *prevp, List_T *nextp, List_T *finalp,
		    int *nindels, bool *plusp, Chrnum_T *chrnum, Chrpos_T *chrpos,
		    Stage3end_T stage3end, int querylength, bool first_read_p, Stage3end_T mate,
		    int hardclip_low, int hardclip_high, bool hide_soft_clips_p) {
  Hittype_T hittype;
  int circularpos;
  int length_querystart, length_queryend;
  List_T substrings_LtoH, junctions_LtoH;
  Substring_T substring_low, substring1, substringN;
  int part_hardclip_low, part_hardclip_high, splice_hardclip1, splice_hardclipN;

  debug3(printf("Entered Cigar_compute_main with %p, hardclip_low %d and hardclip_high %d\n",
		stage3end,hardclip_low,hardclip_high));

  *cigar_fp = Filestring_new(/*id*/0);
  *md_fp = Filestring_new(/*id*/0);
  *nmismatches_refdiff = *nmismatches_bothdiff = 0;
  *nindels = 0;
  *prevp = *nextp = *finalp = (List_T) NULL;

  if (stage3end == NULL) {
    debug3(printf("stage3end is NULL\n"));
    FPRINTF(*cigar_fp,"*");
    part_hardclip_low = hardclip_low;
    part_hardclip_high = hardclip_high;
    *plusp = true;
    *chrnum = 0;
    *chrpos = 0;

  } else if ((hittype = Stage3end_hittype(stage3end)) == TRANSLOC_SPLICE ||
	     (hittype == SAMECHR_SPLICE && merge_samechr_p == false)) {
    debug3(printf("Have a translocation with distant_splice_i %d\n",
		  Stage3end_distant_splice_i(stage3end)));

    if (mate == NULL) {
      /* No notion of concordance, so pick longer side to be main */
      debug3(printf("Mate is NULL, so pick longer side\n"));

      splice_hardclip1 = Stage3end_splice_hardclip1(stage3end);
      splice_hardclipN = Stage3end_splice_hardclipN(stage3end);
      debug3(printf("Computing splice hardclips as %d and %d\n",splice_hardclip1,splice_hardclipN));

      if (splice_hardclip1 < splice_hardclipN) {
	/* Print the longer end (N) as main */
	substringN = Stage3end_substringN(stage3end);
	debug3(printf("substringN has plusp %d\n",Substring_plusp(substringN)));

	if ((*plusp = Substring_plusp(substringN)) == true) {
	  substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	  junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	  part_hardclip_low = splice_hardclip1;
	  part_hardclip_high = 0;
	} else {
	  substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	  junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	  part_hardclip_low = 0;
	  part_hardclip_high = splice_hardclip1;
	}

	print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
			 &substring_low,&(*prevp),&(*nextp),&(*finalp),
			 *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
			 querylength,part_hardclip_low,part_hardclip_high);
	*chrnum = Substring_chrnum(substring_low);
	*chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);

      } else {
	/* Print the longer end (1) as main */
	substring1 = Stage3end_substring1(stage3end);
	debug3(printf("substring1 has plusp %d\n",Substring_plusp(substring1)));

	if ((*plusp = Substring_plusp(substring1)) == true) {
	  substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	  junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	  part_hardclip_low = 0;
	  part_hardclip_high = splice_hardclipN;
	} else {
	  substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	  junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	  part_hardclip_low = splice_hardclipN;
	  part_hardclip_high = 0;
	}

	print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
			 &substring_low,&(*prevp),&(*nextp),&(*finalp),
			 *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
			 querylength,part_hardclip_low,part_hardclip_high);
	*chrnum = Substring_chrnum(substring_low);
	*chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);
      }

    } else if (first_read_p == true) {
      /* N side is concordant, so treat as main */
      splice_hardclip1 = Stage3end_splice_hardclip1(stage3end);
      debug3(printf("first read so N side is concordant.  Splice hardclip1 is %d\n",splice_hardclip1));
      substringN = Stage3end_substringN(stage3end);

      if ((*plusp = Substring_plusp(substringN)) == true) {
	substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	part_hardclip_low = splice_hardclip1;
	part_hardclip_high = 0;
      } else {
	substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	part_hardclip_low = 0;
	part_hardclip_high = splice_hardclip1;
      }

      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
		       querylength,part_hardclip_low,part_hardclip_high);
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);

    } else {
      /* 1 side is concordant, so treat as main */
      splice_hardclipN = Stage3end_splice_hardclipN(stage3end);
      debug3(printf("second read so 1 side is concordant.  Splice hardclipN is %d\n",splice_hardclipN));
      substring1 = Stage3end_substring1(stage3end);

      if ((*plusp = Substring_plusp(substring1)) == true) {
	debug3(printf("plusp based on substring1 is %d\n",*plusp));
	substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	part_hardclip_low = 0;
	part_hardclip_high = splice_hardclipN;
      } else {
	debug3(printf("plusp based on substring1 is %d\n",*plusp));
	substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	part_hardclip_low = splice_hardclipN;
	part_hardclip_high = 0;
      }

      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
		       querylength,part_hardclip_low,part_hardclip_high);
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);
    }

  } else if ((circularpos = Stage3end_circularpos(stage3end)) <= 0) {
    /* Normal alignment */
    *plusp = Stage3end_plusp(stage3end);
    print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		     &substring_low,&(*prevp),&(*nextp),&(*finalp),
		     *cigar_fp,*md_fp,Stage3end_substrings_LtoH(stage3end),
		     Stage3end_junctions_LtoH(stage3end),*plusp,
		     querylength,hardclip_low,hardclip_high);
#if 1
    part_hardclip_low = hardclip_low;
    part_hardclip_high = hardclip_high;
    *chrnum = Substring_chrnum(substring_low);
    *chrpos = Substring_compute_chrpos(substring_low,hardclip_low,hide_soft_clips_p);
#else
    /* Always want the substring with the lowest coordinate */
    if ((substring = Stage3end_substring_low(stage3end,hardclip_low)) == NULL) {
      part_hardclip_low = part_hardclip_high = 0;
      *plusp = true;
      *chrnum = 0;
      *chrpos = 0;
    } else {
      part_hardclip_low = hardclip_low;
      part_hardclip_high = hardclip_high;
      *plusp = Substring_plusp(substring);
      *chrnum = Substring_chrnum(substring);
      *chrpos = Substring_compute_chrpos(substring,hardclip_low,/*hardclip_high,*/hide_soft_clips_p);
    }
#endif

  } else {
    /* Circular alignment: Pick longer side to be the main */
    debug3(printf("Have a circular alignment with circularpos %d\n",circularpos));
    debug3(printf("For length_querystart, will enforce hardclips of %d and %d\n",
		  hardclip_low,querylength-circularpos));
    debug3(printf("For length_queryend, will enforce hardclips of %d and %d\n",
		  circularpos,hardclip_high));

    length_querystart = Cigar_length_substrings(stage3end,querylength,
						hardclip_low,/*hardclip_high*/querylength-circularpos);
    length_queryend = Cigar_length_substrings(stage3end,querylength,
					      /*hardclip_low*/circularpos,hardclip_high);
    debug3(printf("length_querystart is %d, length_queryend is %d\n",length_querystart,length_queryend));

    if (length_querystart > length_queryend) {
      debug3(printf("querystart is longer\n"));
      *plusp = Stage3end_plusp(stage3end);
      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,Stage3end_substrings_LtoH(stage3end),
		       Stage3end_junctions_LtoH(stage3end),*plusp,
		       querylength,hardclip_low,/*hardclip_high*/querylength-circularpos);
#if 1
      part_hardclip_low = hardclip_low;
      part_hardclip_high = querylength - circularpos;
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,hardclip_low,hide_soft_clips_p);
      if (*chrpos > Substring_chrlength(substring_low)) {
	*chrpos -= Substring_chrlength(substring_low);
      }
#else
      if ((substring = Stage3end_substring_low(stage3end,hardclip_low)) == NULL) {
	part_hardclip_low = part_hardclip_high = 0;
	*plusp = true;
	*chrnum = 0;
	*chrpos = 0;
      } else {
	part_hardclip_low = hardclip_low;
	part_hardclip_high = querylength - circularpos;
	*plusp = Substring_plusp(substring);
	*chrnum = Substring_chrnum(substring);
	*chrpos = Substring_compute_chrpos(substring,hardclip_low,
					   /*hardclip_high:querylength-circularpos,*/
					   hide_soft_clips_p);
	if (*chrpos > Substring_chrlength(substring)) {
	  *chrpos -= Substring_chrlength(substring);
	}
      }
#endif

    } else {
      debug3(printf("queryend is longer\n"));
      *plusp = Stage3end_plusp(stage3end);
      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,Stage3end_substrings_LtoH(stage3end),
		       Stage3end_junctions_LtoH(stage3end),*plusp,
		       querylength,/*hardclip_low*/circularpos,hardclip_high);
#if 1
      part_hardclip_low = circularpos;
      part_hardclip_high = hardclip_high;
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,circularpos,hide_soft_clips_p);
      if (*chrpos > Substring_chrlength(substring_low)) {
	*chrpos -= Substring_chrlength(substring_low);
      }
#else
      if ((substring = Stage3end_substring_low(stage3end,/*hardclip_low*/circularpos)) == NULL) {
	part_hardclip_low = part_hardclip_high = 0;
	*plusp = true;
	*chrnum = 0;
	*chrpos = 0;
      } else {
	part_hardclip_low = circularpos;
	part_hardclip_high = hardclip_high;
	*plusp = Substring_plusp(substring);
	*chrnum = Substring_chrnum(substring);
	*chrpos = Substring_compute_chrpos(substring,/*hardclip_low*/circularpos,
					   /*hardclip_high,*/hide_soft_clips_p);
	if (*chrpos > Substring_chrlength(substring)) {
	  *chrpos -= Substring_chrlength(substring);
	}
      }
#endif
    }
  }
  debug3(printf("\n\n"));

  if (sam_hardclip_use_S_p == false) {
    *sequence_hardclip_low = part_hardclip_low;
    *sequence_hardclip_high = part_hardclip_high;
  } else {
    *sequence_hardclip_low = *sequence_hardclip_high = 0;
  }

  debug3(printf("Exiting Cigar_compute_main with plusp %d, chrnum %d, chrpos %u\n",*plusp,*chrnum,*chrpos));
  Filestring_stringify(*cigar_fp);
  Filestring_stringify(*md_fp);

  return;
}


void
Cigar_compute_supplemental (Filestring_T *cigar_fp, Filestring_T *md_fp,
			    int *nmismatches_refdiff, int *nmismatches_bothdiff,
			    int *sequence_hardclip_low, int *sequence_hardclip_high,
			    List_T *prevp, List_T *nextp, List_T *finalp,
			    int *nindels, bool *plusp, Chrnum_T *chrnum, Chrpos_T *chrpos,
			    Stage3end_T stage3end, int querylength, bool first_read_p, Stage3end_T mate,
			    int hardclip_low, int hardclip_high, bool hide_soft_clips_p) {
  Hittype_T hittype;
  int circularpos;
  int length_querystart, length_queryend;
  List_T substrings_LtoH, junctions_LtoH;
  Substring_T substring_low, substring1, substringN;
  int part_hardclip_low, part_hardclip_high, splice_hardclip1, splice_hardclipN;

  debug4(printf("Entered Cigar_compute_supplemental with hardclip_low %d and hardclip_high %d\n",
		hardclip_low,hardclip_high));

  *cigar_fp = Filestring_new(/*id*/0);
  *md_fp = Filestring_new(/*id*/0);
  *nmismatches_refdiff = *nmismatches_bothdiff = 0;
  *nindels = 0;
  *prevp = *nextp = *finalp = (List_T) NULL;

  if (stage3end == NULL) {
    /* Shouldn't be calling for supplemental */
    abort();

  } else if ((hittype = Stage3end_hittype(stage3end)) == TRANSLOC_SPLICE ||
	     (hittype == SAMECHR_SPLICE && merge_samechr_p == false)) {
    debug4(printf("Have a translocation\n"));

    if (mate == NULL) {
      /* No notion of concordance, so pick shorter side to be supplemental */
      debug4(printf("Mate is NULL, so pick shorter side\n"));

      splice_hardclip1 = Stage3end_splice_hardclip1(stage3end);
      splice_hardclipN = Stage3end_splice_hardclipN(stage3end);
      if (splice_hardclip1 < splice_hardclipN) {
	/* Print the shorter end (1) as supplemental */
	substring1 = Stage3end_substring1(stage3end);
	if ((*plusp = Substring_plusp(substring1)) == true) {
	  substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	  junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	  part_hardclip_low = 0;
	  part_hardclip_high = splice_hardclipN;
	} else {
	  substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	  junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	  part_hardclip_low = splice_hardclipN;
	  part_hardclip_high = 0;
	}

	print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
			 &substring_low,&(*prevp),&(*nextp),&(*finalp),
			 *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
			 querylength,part_hardclip_low,part_hardclip_high);
	*chrnum = Substring_chrnum(substring_low);
	*chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);

      } else {
	/* Print the shorter end (N) as supplemental */
	substringN = Stage3end_substringN(stage3end);
	if ((*plusp = Substring_plusp(substringN)) == true) {
	  substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	  junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	  part_hardclip_low = splice_hardclip1;
	  part_hardclip_high = 0;
	} else {
	  substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	  junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	  part_hardclip_low = 0;
	  part_hardclip_high = splice_hardclip1;
	}

	print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
			 &substring_low,&(*prevp),&(*nextp),&(*finalp),
			 *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
			 querylength,part_hardclip_low,part_hardclip_high);
	*chrnum = Substring_chrnum(substring_low);
	*chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);
      }

    } else if (first_read_p == true) {
      /* 1 side is not concordant, so treat as supplemental */
      splice_hardclipN = Stage3end_splice_hardclipN(stage3end);
      substring1 = Stage3end_substring1(stage3end);
      if ((*plusp = Substring_plusp(substring1)) == true) {
	substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	part_hardclip_low = 0;
	part_hardclip_high = splice_hardclipN;
      } else {
	substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	part_hardclip_low = splice_hardclipN;
	part_hardclip_high = 0;
      }

      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
		       querylength,part_hardclip_low,part_hardclip_high);
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);

    } else {
      /* N side is not concordant, so treat as supplemental */
      splice_hardclip1 = Stage3end_splice_hardclip1(stage3end);
      substringN = Stage3end_substringN(stage3end);
      if ((*plusp = Substring_plusp(substringN)) == true) {
	substrings_LtoH = Stage3end_substrings_1toN(stage3end);
	junctions_LtoH = Stage3end_junctions_1toN(stage3end);
	part_hardclip_low = splice_hardclip1;
	part_hardclip_high = 0;
      } else {
	substrings_LtoH = Stage3end_substrings_Nto1(stage3end);
	junctions_LtoH = Stage3end_junctions_Nto1(stage3end);
	part_hardclip_low = 0;
	part_hardclip_high = splice_hardclip1;
      }

      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,substrings_LtoH,junctions_LtoH,*plusp,
		       querylength,part_hardclip_low,part_hardclip_high);
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,part_hardclip_low,hide_soft_clips_p);
    }

  } else if ((circularpos = Stage3end_circularpos(stage3end)) <= 0) {
    /* Normal alignment: Shouldn't be calling for supplemental */
    abort();

  } else {
    /* Circular alignment: Pick shorter side to be the supplemental */
    debug4(printf("Have a circular alignment with circularpos %d\n",circularpos));
    debug4(printf("For length_querystart, will enforce hardclips of %d and %d\n",
		  hardclip_low,querylength-circularpos));
    debug4(printf("For length_queryend, will enforce hardclips of %d and %d\n",
		  circularpos,hardclip_high));

    length_querystart = Cigar_length_substrings(stage3end,querylength,
						hardclip_low,/*hardclip_high*/querylength-circularpos);
    length_queryend = Cigar_length_substrings(stage3end,querylength,
					      /*hardclip_low*/circularpos,hardclip_high);
    debug4(printf("length_querystart is %d, length_queryend is %d\n",length_querystart,length_queryend));

    if (length_querystart > length_queryend) {
      debug4(printf("queryend is shorter\n"));
      *plusp = Stage3end_plusp(stage3end);
      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,Stage3end_substrings_LtoH(stage3end),
		       Stage3end_junctions_LtoH(stage3end),*plusp,
		       querylength,/*hardclip_low*/circularpos,hardclip_high);
#if 1
      part_hardclip_low = circularpos;
      part_hardclip_high = hardclip_high;
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,circularpos,hide_soft_clips_p);
      if (*chrpos > Substring_chrlength(substring_low)) {
	*chrpos -= Substring_chrlength(substring_low);
      }
#else
      if ((substring = Stage3end_substring_low(stage3end,/*hardclip_low*/circularpos)) == NULL) {
	part_hardclip_low = part_hardclip_high = 0;
	*plusp = true;
	*chrnum = 0;
	*chrpos = 0;
      } else {
	part_hardclip_low = circularpos;
	part_hardclip_high = hardclip_high;
	*plusp = Substring_plusp(substring);
	*chrnum = Substring_chrnum(substring);
	*chrpos = Substring_compute_chrpos(substring,/*hardclip_low*/circularpos,
					   /*hardclip_high,*/hide_soft_clips_p);
	if (*chrpos > Substring_chrlength(substring)) {
	  *chrpos -= Substring_chrlength(substring);
	}
      }
#endif

    } else {
      debug4(printf("querystart is shorter\n"));
      *plusp = Stage3end_plusp(stage3end);
      print_substrings(&(*nindels),&(*nmismatches_refdiff),&(*nmismatches_bothdiff),
		       &substring_low,&(*prevp),&(*nextp),&(*finalp),
		       *cigar_fp,*md_fp,Stage3end_substrings_LtoH(stage3end),
		       Stage3end_junctions_LtoH(stage3end),*plusp,
		       querylength,hardclip_low,/*hardclip_high*/querylength-circularpos);
#if 1
      part_hardclip_low = hardclip_low;
      part_hardclip_high = querylength - circularpos;
      *chrnum = Substring_chrnum(substring_low);
      *chrpos = Substring_compute_chrpos(substring_low,hardclip_low,hide_soft_clips_p);
      if (*chrpos > Substring_chrlength(substring_low)) {
	*chrpos -= Substring_chrlength(substring_low);
      }
#else
      if ((substring = Stage3end_substring_low(stage3end,hardclip_low)) == NULL) {
	part_hardclip_low = part_hardclip_high = 0;
	*plusp = true;
	*chrnum = 0;
	*chrpos = 0;
      } else {
	part_hardclip_low = hardclip_low;
	part_hardclip_high = querylength - circularpos;
	*plusp = Substring_plusp(substring);
	*chrnum = Substring_chrnum(substring);
	*chrpos = Substring_compute_chrpos(substring,hardclip_low,
					   /*hardclip_high:querylength-circularpos,*/
					   hide_soft_clips_p);
	if (*chrpos > Substring_chrlength(substring)) {
	  *chrpos -= Substring_chrlength(substring);
	}
      }
#endif
    }
  }
  debug4(printf("\n\n"));

  if (sam_hardclip_use_S_p == false) {
    *sequence_hardclip_low = part_hardclip_low;
    *sequence_hardclip_high = part_hardclip_high;
  } else {
    *sequence_hardclip_low = *sequence_hardclip_high = 0;
  }

  debug4(printf("Exiting Cigar_compute_supplemental with plusp %d, chrnum %d, chrpos %u\n",*plusp,*chrnum,*chrpos));
  Filestring_stringify(*cigar_fp);
  Filestring_stringify(*md_fp);

  return;
}


