/***************************************************************************
 *   Copyright (C) 2009 by Kai Dombrowe <just89@gmx.de>                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "pgame.h"
#include "listwidget.h"
#include "listwidgetitem.h"


#include <QGraphicsLinearLayout>
#include <KLocale>
#include <KConfig>
#include <KConfigGroup>
#include <QAction>
#include <KDebug>
#include <KProcess>
#include <KStandardDirs>
#include <KMessageBox>
#include <KServiceGroup>
#include <KService>


PGame::PGame(QObject *parent, const QVariantList &args)
        : Plasma::PopupApplet(parent, args),
        m_currentItem(0)
{

    setAspectRatioMode(Plasma::IgnoreAspectRatio);
    setBackgroundHints(DefaultBackground);
    setHasConfigurationInterface(true);
    setPopupIcon(KIcon("applications-games"));
    resize(200, 200);

}


PGame::~PGame()
{

    saveConfig();

}


void PGame::init()
{

    m_widget = new QGraphicsWidget(this);
    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(m_widget);

    Plasma::ScrollWidget *scrollWidget = new Plasma::ScrollWidget;
    m_listWidget = new ListWidget(m_widget);
    connect(m_listWidget, SIGNAL(droped(ListWidgetItem*)), this, SLOT(itemDroped(ListWidgetItem*)));
    scrollWidget->setWidget(m_listWidget);

    layout->addItem(scrollWidget);
    m_widget->setLayout(layout);

    loadConfig();
    Plasma::PopupApplet::init();

}

   
QGraphicsWidget *PGame::graphicsWidget()
{

    return m_widget;

}


void PGame::showGameDialog()
{

    KDialog *dialog = new KDialog;
    QWidget *mainWidget = new QWidget(dialog);
    ui_gameDialog.setupUi(mainWidget);
    dialog->setMainWidget(mainWidget);
    dialog->setAttribute(Qt::WA_DeleteOnClose);
    connect(dialog, SIGNAL(okClicked()), this, SLOT(createGame()));
    connect(dialog, SIGNAL(okClicked()), dialog, SLOT(close()));
    dialog->show();

}


void PGame::createGame()
{

    ListWidgetItem *item = new ListWidgetItem(this);
    connect(item, SIGNAL(edit()), this, SLOT(editGame()));
    connect(item, SIGNAL(remove()), this, SLOT(removeGame()));
    connect(item, SIGNAL(run()), this, SLOT(runGame()));
    item->setTitle(ui_gameDialog.titleEdit->text());
    item->setIcon(ui_gameDialog.iconButton->icon());
    item->setRating(ui_gameDialog.ratingInput->value());
    item->setX(ui_gameDialog.xCheck->isChecked());
    item->setCommand(ui_gameDialog.commandRequester->text());
    m_listWidget->addItem(item);
        
}


void PGame::editGame()
{

    ListWidgetItem *item = static_cast<ListWidgetItem*>(sender());
    if (!item)
        return;
        
    m_currentItem = item;
    showGameDialog();
    KDialog *dialog = static_cast<KDialog*>(ui_gameDialog.titleEdit->parentWidget()->parentWidget());
    dialog->disconnect();
    connect(dialog, SIGNAL(okClicked()), this, SLOT(gameChanged()));
    connect(dialog, SIGNAL(okClicked()), dialog, SLOT(close()));
    
    ui_gameDialog.titleEdit->setText(item->title());
    ui_gameDialog.iconButton->setIcon(item->icon());
    ui_gameDialog.ratingInput->setValue(item->rating());
    ui_gameDialog.xCheck->setChecked(item->X());
    ui_gameDialog.commandRequester->setUrl(item->command());
    
}


void PGame::gameChanged()
{

    if (!m_currentItem)
        return;
        
    m_currentItem->setTitle(ui_gameDialog.titleEdit->text());
    m_currentItem->setIcon(ui_gameDialog.iconButton->icon());
    m_currentItem->setRating(ui_gameDialog.ratingInput->value());
    m_currentItem->setX(ui_gameDialog.xCheck->isChecked());
    m_currentItem->setCommand(ui_gameDialog.commandRequester->text());
    m_currentItem = 0;
    
}


void PGame::removeGame()
{

    ListWidgetItem *item = dynamic_cast<ListWidgetItem*>(sender());
    if (!item) {
        item = m_listWidget->selectedItem();
        if (!item) {
            return;
        }
    }
    kDebug() << "removing >>" << item->title();
    m_listWidget->removeItem(item);

}


void PGame::saveConfig()
{

    QList<ListWidgetItem*> items = m_listWidget->items();
    KConfigGroup cfg = config();

    cfg.writeEntry("Games", items.size());
    for (int i = 0; i < items.size(); i++) {
        ListWidgetItem *item = items.at(i);
        cfg.writeEntry(QString("Game %1 Title").arg(i), item->title());
        cfg.writeEntry(QString("Game %1 Icon").arg(i), item->icon());
        cfg.writeEntry(QString("Game %1 Rating").arg(i), item->rating());
        cfg.writeEntry(QString("Game %1 X").arg(i), item->X());
        cfg.writeEntry(QString("Game %1 Command").arg(i), item->command());
    }

}


void PGame::loadConfig()
{

    KConfigGroup cfg = config();

    int games = cfg.readEntry("Games", -1);
    
    if (games == -1) {
        firstStart();
    } else {
        for (int i = 0; i < games; i++) {
            ListWidgetItem *item = new ListWidgetItem(m_listWidget);
            item->setTitle(cfg.readEntry(QString("Game %1 Title").arg(i), "No Title"));
            item->setIcon(cfg.readEntry(QString("Game %1 Icon").arg(i), QString()));
            item->setRating(cfg.readEntry(QString("Game %1 Rating").arg(i), 0));
            item->setX(cfg.readEntry(QString("Game %1 X").arg(i), false));
            item->setCommand(cfg.readEntry(QString("Game %1 Command").arg(i), QString()));
            connect(item, SIGNAL(edit()), this, SLOT(editGame()));
            connect(item, SIGNAL(remove()), this, SLOT(removeGame()));
            connect(item, SIGNAL(run()), this, SLOT(runGame()));
            m_listWidget->addItem(item);
        }
    }

}


QList<QAction*> PGame::contextualActions()
{

    QList<QAction*> list;
    
    QAction *add = new QAction(this);
    add->setText(i18n("Add new Game"));
    add->setIcon(KIcon("list-add"));
    
    connect(add, SIGNAL(triggered()), this, SLOT(showGameDialog()));
    
    QAction *remove = new QAction(this);
    remove->setText(i18n("Remove selected Game"));
    remove->setIcon(KIcon("list-remove"));
    if (!m_listWidget->selectedItem())
        remove->setDisabled(true);
    connect(remove, SIGNAL(triggered()), this, SLOT(removeGame()));
    
    QAction *separator = new QAction(this);
    separator->setSeparator(true);
    
    list.append(add);
    list.append(remove);
    list.append(separator);
    list.append(Plasma::PopupApplet::contextualActions());
    
    return list;

}


void PGame::runGame()
{

    ListWidgetItem *item = dynamic_cast<ListWidgetItem*>(sender());
    if (!item) {
        item = m_listWidget->selectedItem();
        if (!item) {
            return;
        }
    }
        
    kDebug() << "starting >>" << item->title();        

    if (!item->X()) {
        KProcess *process = new KProcess(this);
        process->setShellCommand(item->command());
        process->start();
        connect(process, SIGNAL(finished(int)), this, SLOT(deleteProcess(int)));
        return;
    }
                                                            
    QString home = QDir::homePath();
    QString cmd  = item->command();

    //xbin + xinitrc
    KUrl xbin = KStandardDirs::findExe("X");
    QFile *rc = new QFile((home+"/.xinitrc_pgame"));
    if (rc->exists())                                  
        rc->remove();                               

    if(!rc->open(QIODevice::WriteOnly|QIODevice::Text)) {                                                        
        KMessageBox::sorry(0, QString("cannot open \"%1\".").arg((home+"/.xinitrc_pgame")),
                        QString(), 0);                                                                 
        return;                                                                                 
    }                                                                                               

    QTextStream out(rc);
    out << QString("exec %1").arg(cmd) << "\n";
    rc->close();
    delete rc;
    
    //xnumber
    QString tmp = getenv("TMPDIR");
    if(!QDir(tmp).exists() || tmp.isEmpty())
        tmp = "/tmp";                                     

    int xnumber = 0;
    while(true) {
        if (xnumber > 99)
            break;
            
        if(QFile((tmp+QString("/.X%1-lock").arg(xnumber))).exists()) {
            xnumber++;                                                              
        } else {                                                                                
            break;
        }
    }       
    
    //start x                                                                                   
    KProcess *process = new KProcess(this);                                                                 
    QString start = QString("xinit %1/.xinitrc_pgame -- %3 :%2 -br" ).arg(home).arg(xnumber).arg(xbin.toLocalFile());
    process->setOutputChannelMode(KProcess::MergedChannels); // debug                                                                                       
    process->setShellCommand(start);                                                                                                               
    kDebug() << "cmd >" << start;                                                                                                              
    process->start();
    connect(process, SIGNAL(finished(int)), this, SLOT(deleteProcess(int)));

}


void PGame::deleteProcess(const int &)
{

    KProcess *process = static_cast<KProcess*>(sender());
    if (process)
        process->deleteLater();

}


void PGame::firstStart()
{

    kDebug() << ">> First Start <<";
    KServiceGroup::Ptr root = KServiceGroup::group("")->group("Games/");
    if (!root || !root->isValid())
        return;
    browse(root);                

}


void PGame::browse(KServiceGroup::Ptr group)
{

    const KServiceGroup::List list = group->entries(true, true, false, false);

    for (KServiceGroup::List::ConstIterator it = list.constBegin(); it != list.constEnd(); ++it) {
        const KSycocaEntry::Ptr ptr = (*it);

        if (ptr->isType(KST_KService)) {
            const KService::Ptr service = KService::Ptr::staticCast(ptr);

            if (service->noDisplay())
                continue;

            ListWidgetItem *item = new ListWidgetItem(m_listWidget);
            connect(item, SIGNAL(edit()), this, SLOT(editGame()));
            connect(item, SIGNAL(remove()), this, SLOT(removeGame()));
            connect(item, SIGNAL(run()), this, SLOT(runGame()));
            item->setIcon(service->icon());
            item->setTitle(service->name());
            item->setCommand(service->exec());
            m_listWidget->addItem(item);
            kDebug() << "add >>" << service->name();
        }
    }
    
    foreach (KServiceGroup::Ptr p, group->groupEntries()) {
        browse(p);
    }

}


void PGame::itemDroped(ListWidgetItem *item)
{

    connect(item, SIGNAL(edit()), this, SLOT(editGame()));
    connect(item, SIGNAL(remove()), this, SLOT(removeGame()));
    connect(item, SIGNAL(run()), this, SLOT(runGame())); 

}


#include "pgame.moc"
